{-----------------------------------------------------------------------------
This Software is placed into Publlic Domain and distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied.
The Initial Developer is Martin Waldenburg
(Martin.Waldenburg@T-Online.de).
-----------------------------------------------------------------------------}
unit mwScrollingView;

interface

uses
  System.Drawing, System.Collections, System.ComponentModel,
  System.Windows.Forms, System.Data, mwCommandProcessor;

type
  TAnchorStyles = System.Windows.Forms.AnchorStyles;
  TBorderStyle = System.Windows.Forms.BorderStyle;
  TRectangle = System.Drawing.Rectangle;
  TScrollBars = System.Windows.Forms.ScrollBars;
  TMouseButtons = System.Windows.Forms.MouseButtons;
  TmwMouseLanded = (mlGutter, mlView);
  
  TmwScrollingView = class;

  TmwScrollingViewCommander = class(TmwCommandProcessor)
  private
    fMouseDownGutterHandler: TmwMouseEventArgsCommand;
    fPaintGutterHandler: TmwPaintEventArgsCommand;
    fPaintHandler: TmwPaintEventArgsCommand;
    fResizeHandler: TmwEventArgsCommand;
    fVValueChangedHandler: TmwEventArgsCommand;
    fHValueChangedHandler: TmwEventArgsCommand;
    procedure set_MouseDownGutterHandler(const Value: TmwMouseEventArgsCommand);
    procedure set_PaintGutterHandler(const Value: TmwPaintEventArgsCommand);
    procedure set_PaintHandler(const Value: TmwPaintEventArgsCommand);
    procedure set_ResizeHandler(const Value: TmwEventArgsCommand);
    procedure set_HValueChangedHandler(const Value: TmwEventArgsCommand);
    procedure set_VValueChangedHandler(const Value: TmwEventArgsCommand);
  public
    procedure OnHValueChanged(Sender: TObject; EA: EventArgs); virtual;
    procedure OnMouseDownGutter(Sender: TObject; MEA: MouseEventArgs); virtual;
    procedure OnPaint(Sender: TObject; PEA: PaintEventArgs); virtual;
    procedure OnPaintGutter(Sender: TObject; PEA: PaintEventArgs); virtual;
    procedure OnResize(Sender: TObject; REA: EventArgs); virtual;
    procedure OnVValueChanged(Sender: TObject; EA: EventArgs); virtual;
  published
    property OnHValueChangedHandler: TmwEventArgsCommand read fHValueChangedHandler write set_HValueChangedHandler;
    property OnMouseDownGutterHandler: TmwMouseEventArgsCommand read fMouseDownGutterHandler write set_MouseDownGutterHandler;
    property OnPaintHandler: TmwPaintEventArgsCommand read fPaintHandler write set_PaintHandler;
    property OnPaintGutterHandler: TmwPaintEventArgsCommand read fPaintGutterHandler write set_PaintGutterHandler;
    property OnResizeHandler: TmwEventArgsCommand read fResizeHandler write set_ResizeHandler;
    property OnVValueChangedHandler: TmwEventArgsCommand read fVValueChangedHandler write set_VValueChangedHandler;
  end;

  TmwHScrollBar = class(System.Windows.Forms.HScrollBar)
  strict protected
    procedure OnMouseDown(e: MouseEventArgs); override;
    procedure OnScroll(e: ScrollEventArgs); override;
    procedure OnValueChanged(e: EventArgs); override;
  public
    function get_MainControl: TmwScrollingView;
    property MainControl: TmwScrollingView read get_MainControl;
  end;

  TmwVScrollBar = class(System.Windows.Forms.VScrollBar)
  strict protected
    procedure OnMouseDown(e: MouseEventArgs); override;
    procedure OnScroll(e: ScrollEventArgs); override;
    procedure OnValueChanged(e: EventArgs); override;
  public
    function get_MainControl: TmwScrollingView;
    property MainControl: TmwScrollingView read get_MainControl;
  end;

  TmwGutter = class(System.Windows.Forms.Control)
  strict protected
    procedure OnMouseDown(e: MouseEventArgs); override;
    procedure OnPaint(e: PaintEventArgs); override;
  public
    function get_MainControl: TmwScrollingView;
    property MainControl: TmwScrollingView read get_MainControl;
  end;

  TmwView = class(System.Windows.Forms.Control)
  strict protected
    procedure OnMouseDown(e: MouseEventArgs); override;
    procedure OnMouseEnter(e: EventArgs); override;
    procedure OnMouseHover(e: EventArgs); override;
    procedure OnMouseLeave(e: EventArgs); override;
    procedure OnMouseMove(e: MouseEventArgs); override;
    procedure OnMouseUp(e: MouseEventArgs); override;
    procedure OnMouseWheel(e: MouseEventArgs); override;
    procedure OnPaint(e: PaintEventArgs); override;
  public
    function get_MainControl: TmwScrollingView;
    property MainControl: TmwScrollingView read get_MainControl;
  end;

  TmwClientArea = class(System.Windows.Forms.Control)
   strict protected
    /// <summary>
    /// </summary>
    Components: System.ComponentModel.Container;
    fCreating: Boolean;
    fGutter: TmwGutter;
    fView: TmwView;
    /// <summary>
    /// </summary>
    procedure InitializeControl;
    /// <summary>
    /// Clean up any resources being used.
    /// </summary>
    procedure Dispose(Disposing: Boolean); override;
    procedure OnResize(e: EventArgs); override;
  protected
    procedure ReAlignChilds;
  public
    constructor Create;
    property Gutter: TmwGutter read fGutter;
    function get_MainControl: TmwScrollingView;
    property MainControl: TmwScrollingView read get_MainControl;
    property View: TmwView read fView;
  end;

  TmwScrollingView = class(System.Windows.Forms.Panel)
  private
  strict protected
    /// <summary>
    /// </summary>
    Components: System.ComponentModel.Container;
    fCreating: Boolean;
    fClientArea: TmwClientArea;
    fCommander: TmwScrollingViewCommander;
    FScrollBars: TScrollBars;
    fHScrollBar: TmwHScrollBar;
    fVScrollBar: TmwVScrollBar;
    fVValueChanged: EventHandler;
    fHValueChanged: EventHandler;
    fVScroll: ScrollEventHandler;
    fHScroll: ScrollEventHandler;
    fPaintGutter: PaintEventHandler;
    fPaintView: PaintEventHandler;
    fMouseDownGutter: MouseEventHandler;
    fMouseDownView: MouseEventHandler;
    fMouseLeaveView: EventHandler;
    fMouseEnterView: EventHandler;
    fMouseHoverView: EventHandler;
    fMouseUpView: MouseEventHandler;
    fMouseMoveView: MouseEventHandler;
    fMouseWheelView: MouseEventHandler;
    /// <summary>
    /// </summary>
    procedure InitializeControl;
    /// <summary>
    /// Clean up any resources being used.
    /// </summary>
    procedure Dispose(Disposing: Boolean); override;
    procedure OnKeyDown(e: KeyEventArgs); override;
    procedure OnKeyPress(e: KeyPressEventArgs); override;
    procedure OnKeyUp(e: KeyEventArgs); override;
    procedure OnMouseDown(e: MouseEventArgs); override;
    procedure OnResize(e: EventArgs); override;
    procedure ReAlignChilds;
  protected
    fMouseLanded: TmwMouseLanded;
    function get_BorderHight: Integer;
    function get_BorderWidth: Integer;
    function get_HScrollBarHeight: Integer;
    function get_VScrollBarWidth: Integer;
    function get_HScrollBarWidth: Integer;
    function get_VScrollBarHeight: Integer;
    function get_ClientAreaHeight: Integer;
    function get_ClientAreaWidth: Integer;
    function get_HScrollBarVisible: Boolean;
    function get_VScrollBarVisible: Boolean;
    function get_ScrollHeight: Integer;
    function get_ScrollTop: Integer;
    function get_ScrollWidth: Integer;
    function get_ScrollLeft: Integer;
    procedure OnHScroll(e: ScrollEventArgs); virtual;
    procedure OnHValueChanged(e: EventArgs); virtual;
    procedure OnMouseDownGutter(e: MouseEventArgs); virtual;
    procedure OnMouseDownView(e: MouseEventArgs); virtual;
    procedure OnMouseEnterView(e: EventArgs); virtual;
    procedure OnMouseHoverView(e: EventArgs); virtual;
    procedure OnMouseLeaveView(e: EventArgs); virtual;
    procedure OnMouseMoveView(e: MouseEventArgs); virtual;
    procedure OnMouseUpView(e: MouseEventArgs); virtual;
    procedure OnMouseWheelView(e: MouseEventArgs); virtual;
    procedure OnPaintGutter(e: PaintEventArgs); virtual;
    procedure OnPaintView(e: PaintEventArgs); virtual;
    procedure OnVScroll(e: ScrollEventArgs); virtual;
    procedure OnVValueChanged(e: EventArgs); virtual;
    property BorderHight: Integer read get_BorderHight;
    property BorderWidth: Integer read get_BorderWidth;
    property ClientAreaHeight: Integer read get_ClientAreaHeight;
    property ClientAreaWidth: Integer read get_ClientAreaWidth;
    property HScrollBarHeight: Integer read get_HScrollBarHeight;
    property HScrollBarVisible: Boolean read get_HScrollBarVisible;
    property HScrollBarWidth: Integer read get_HScrollBarWidth;
    property ScrollHeight: Integer read get_ScrollHeight;
    property ScrollWidth: Integer read get_ScrollWidth;
    property ScrollLeft: Integer read get_ScrollLeft;
    property ScrollTop: Integer read get_ScrollTop;
    property VScrollBarHeight: Integer read get_VScrollBarHeight;
    property VScrollBarVisible: Boolean read get_VScrollBarVisible;
    property VScrollBarWidth: Integer read get_VScrollBarWidth;
  public
    constructor Create;
    function get_Gutter: TmwGutter;
    function get_View: TmwView;
    property ClientArea: TmwClientArea read fClientArea;
    property Gutter: TmwGutter read get_Gutter;
    property HScrollBar: TmwHScrollBar read fHScrollBar;
    property MouseLanded: TmwMouseLanded read fMouseLanded;
    property View: TmwView read get_View;
    property VScrollBar: TmwVScrollBar read fVScrollBar;
  published
    function get_GutterWidth: Integer;
    procedure set_Commander(const Value: TmwScrollingViewCommander);
    procedure set_GutterWidth(const Value: Integer);
    procedure set_ScrollBars(const Value: TScrollBars);
    procedure set_HScroll(const Value: ScrollEventHandler);
    procedure set_HValueChanged(const Value: EventHandler);
    procedure set_MouseDownGutter(const Value: MouseEventHandler);
    procedure set_MouseDownView(const Value: MouseEventHandler);
    procedure set_MouseEnterView(const Value: EventHandler);
    procedure set_MouseHoverView(const Value: EventHandler);
    procedure set_MouseLeaveView(const Value: EventHandler);
    procedure set_MouseMoveView(const Value: MouseEventHandler);
    procedure set_MouseUpView(const Value: MouseEventHandler);
    procedure set_MouseWheelView(const Value: MouseEventHandler);
    procedure set_PaintGutter(const Value: PaintEventHandler);
    procedure set_PaintView(const Value: PaintEventHandler);
    procedure set_VScroll(const Value: ScrollEventHandler);
    procedure set_VValueChanged(const Value: EventHandler);
    property Commander: TmwScrollingViewCommander read fCommander write set_Commander;
    property GutterWidth: Integer read get_GutterWidth write set_GutterWidth;
    property HScroll: ScrollEventHandler read fHScroll write set_HScroll;
    property HValueChanged: EventHandler read fHValueChanged write set_HValueChanged;
    property MouseDownGutter: MouseEventHandler read fMouseDownGutter write set_MouseDownGutter;
    property MouseDownView: MouseEventHandler read fMouseDownView write set_MouseDownView;
    property MouseEnterView: EventHandler read fMouseEnterView write set_MouseEnterView;
    property MouseHoverView: EventHandler read fMouseHoverView write set_MouseHoverView;
    property MouseLeaveView: EventHandler read fMouseLeaveView write set_MouseLeaveView;
    property MouseMoveView: MouseEventHandler read fMouseMoveView write set_MouseMoveView;
    property MouseUpView: MouseEventHandler read fMouseUpView write set_MouseUpView;
    property MouseWheelView: MouseEventHandler read fMouseWheelView write set_MouseWheelView;
    property PaintGutter: PaintEventHandler read fPaintGutter write set_PaintGutter;
    property PaintView: PaintEventHandler read fPaintView write set_PaintView;
    property ScrollBars: TScrollBars read FScrollBars write set_ScrollBars;
    property VScroll: ScrollEventHandler read fVScroll write set_VScroll;
    property VValueChanged: EventHandler read fVValueChanged write set_VValueChanged;
  end;

  [assembly: RuntimeRequiredAttribute(TypeOf(TmwScrollingView))]

implementation

uses
  System.Globalization;

{ TmwScrollingViewCommander }

procedure TmwScrollingViewCommander.OnHValueChanged(Sender: TObject; EA: EventArgs);
begin
  if fHValueChangedHandler <> nil then
    fHValueChangedHandler.Execute(Sender, EA);
end;

procedure TmwScrollingViewCommander.OnMouseDownGutter(Sender: TObject; MEA: MouseEventArgs);
begin
  if fMouseDownGutterHandler <> nil then
    fMouseDownGutterHandler.Execute(Sender, MEA);
end;

procedure TmwScrollingViewCommander.OnPaint(Sender: TObject; PEA: PaintEventArgs);
begin
  if fPaintHandler <> nil then
    fPaintHandler.Execute(Sender, PEA);
end;

procedure TmwScrollingViewCommander.OnPaintGutter(Sender: TObject; PEA: PaintEventArgs);
begin
  if fPaintGutterHandler <> nil then
    fPaintGutterHandler.Execute(Sender, PEA);
end;

procedure TmwScrollingViewCommander.OnResize(Sender: TObject; REA: EventArgs);
begin
  if fResizeHandler <> nil then
    fResizeHandler.Execute(Sender, REA);
end;

procedure TmwScrollingViewCommander.OnVValueChanged(Sender: TObject; EA: EventArgs);
begin
  if fVValueChangedHandler <> nil then
    fVValueChangedHandler.Execute(Sender, EA);
end;

procedure TmwScrollingViewCommander.set_HValueChangedHandler(const Value: TmwEventArgsCommand);
begin
  case fHValueChangedHandler <> nil of
    True:
      begin
        CommandTable.Remove(fHValueChangedHandler);
        fHValueChangedHandler := Value;
        AddCommand(Value);
      end;
    False:
      begin
        fHValueChangedHandler := Value;
        AddCommand(Value);
      end;
  end;
end;

procedure TmwScrollingViewCommander.set_MouseDownGutterHandler(const Value: TmwMouseEventArgsCommand);
begin
  case fMouseDownGutterHandler <> nil of
    True:
      begin
        CommandTable.Remove(fMouseDownGutterHandler);
        fMouseDownGutterHandler := Value;
        AddCommand(Value);
      end;
    False:
      begin
        fMouseDownGutterHandler := Value;
        AddCommand(Value);
      end;
  end;
end;

procedure TmwScrollingViewCommander.set_PaintGutterHandler(const Value: TmwPaintEventArgsCommand);
begin
  case fPaintGutterHandler <> nil of
    True:
      begin
        CommandTable.Remove(fPaintGutterHandler);
        fPaintGutterHandler := Value;
        AddCommand(Value);
      end;
    False:
      begin
        fPaintGutterHandler := Value;
        AddCommand(Value);
      end;
  end;
end;

procedure TmwScrollingViewCommander.set_PaintHandler(const Value: TmwPaintEventArgsCommand);
begin
  case fPaintHandler <> nil of
    True:
      begin
        CommandTable.Remove(fPaintHandler);
        fPaintHandler := Value;
        AddCommand(Value);
      end;
    False:
      begin
        fPaintHandler := Value;
        AddCommand(Value);
      end;
  end;
end;

procedure TmwScrollingViewCommander.set_ResizeHandler(const Value: TmwEventArgsCommand);
begin
  case fResizeHandler <> nil of
    True:
      begin
        CommandTable.Remove(fResizeHandler);
        fResizeHandler := Value;
        AddCommand(Value);
      end;
    False:
      begin
        fResizeHandler := Value;
        AddCommand(Value);
      end;
  end;
end;

procedure TmwScrollingViewCommander.set_VValueChangedHandler(const Value: TmwEventArgsCommand);
begin
  case fVValueChangedHandler <> nil of
    True:
      begin
        CommandTable.Remove(fVValueChangedHandler);
        fVValueChangedHandler := Value;
        AddCommand(Value);
      end;
    False:
      begin
        fVValueChangedHandler := Value;
        AddCommand(Value);
      end;
  end;
end;

{ TmwHScrollBar }

function TmwHScrollBar.get_MainControl: TmwScrollingView;
begin
  Result:= (Self.Parent as TmwScrollingView);
end;

procedure TmwHScrollBar.OnMouseDown(e: MouseEventArgs);
begin
  MainControl.Focus;
  inherited OnMouseDown(e);
end;

procedure TmwHScrollBar.OnScroll(e: ScrollEventArgs);
begin
  MainControl.OnHScroll(e);
end;

procedure TmwHScrollBar.OnValueChanged(e: EventArgs);
begin
  MainControl.ClientArea.ReAlignChilds;
  MainControl.OnHValueChanged(e);
end;

{ TmwVScrollBar }

function TmwVScrollBar.get_MainControl: TmwScrollingView;
begin
  Result:= (Self.Parent as TmwScrollingView);
end;

procedure TmwVScrollBar.OnMouseDown(e: MouseEventArgs);
begin
  MainControl.Focus;
  inherited OnMouseDown(e);
end;

procedure TmwVScrollBar.OnScroll(e: ScrollEventArgs);
begin
  MainControl.OnVScroll(e);
end;

procedure TmwVScrollBar.OnValueChanged(e: EventArgs);
begin
  MainControl.ClientArea.ReAlignChilds;
  MainControl.OnVValueChanged(e);
end;

{ TmwGutter }

function TmwGutter.get_MainControl: TmwScrollingView;
begin
  Result:= (Self.Parent.Parent as TmwScrollingView);
end;

procedure TmwGutter.OnMouseDown(e: MouseEventArgs);
begin
  MainControl.fMouseLanded:= mlGutter;
  MainControl.Focus();
  case e.Button of
    TMouseButtons.Left: MainControl.View.Capture := True;
  end;
  MainControl.OnMouseDownGutter(e);
end;

procedure TmwGutter.OnPaint(e: PaintEventArgs);
begin
  MainControl.OnPaintGutter(e);
end;

{ TmwView }

function TmwView.get_MainControl: TmwScrollingView;
begin
  Result:= (Self.Parent.Parent as TmwScrollingView);
end;

procedure TmwView.OnMouseDown(e: MouseEventArgs);
begin
  MainControl.fMouseLanded:= mlView;
  MainControl.Focus();
  case e.Button of
    TMouseButtons.Left: Self.Capture := True;
  end;
  MainControl.OnMouseDownView(e);
end;

procedure TmwView.OnMouseEnter(e: EventArgs);
begin
  MainControl.OnMouseEnterView(e);
end;

procedure TmwView.OnMouseHover(e: EventArgs);
begin
  MainControl.OnMouseHoverView(e);
end;

procedure TmwView.OnMouseLeave(e: EventArgs);
begin
  MainControl.OnMouseLeaveView(e);
end;

procedure TmwView.OnMouseMove(e: MouseEventArgs);
begin
  MainControl.OnMouseMoveView(e);
end;

procedure TmwView.OnMouseUp(e: MouseEventArgs);
begin
  case e.Button of
    TMouseButtons.Left: Self.Capture := False;
  end;
  MainControl.OnMouseUpView(e);
end;

procedure TmwView.OnMouseWheel(e: MouseEventArgs);
begin
  MainControl.OnMouseWheelView(e);
end;

procedure TmwView.OnPaint(e: PaintEventArgs);
begin
  MainControl.OnPaintView(e);
end;

{ TmwClientArea }

{$REGION 'Initialization of TmwClientArea'}
/// <summary>
/// </summary>
procedure TmwClientArea.InitializeControl;
begin
  Self.SuspendLayout;
  Self.fView := TmwView.Create;
  Self.fGutter := TmwGutter.Create;
  //
  // fView
  //
  Self.fView.Anchor := System.Windows.Forms.AnchorStyles.None;
  Self.fView.BackColor := System.Drawing.SystemColors.Window;
  Self.fView.Location := System.Drawing.Point.Create(30, 0);
  Self.fView.Name := 'fView';
  Self.fView.Size := System.Drawing.Size.Create(140, 230);
  Self.fView.TabIndex := 0;
//  Self.fView.Parent:= Self;
  //
  // fGutter
  // 
  Self.fGutter.Anchor := System.Windows.Forms.AnchorStyles.Left;
  Self.fGutter.Location := System.Drawing.Point.Create(0, 0);
  Self.fGutter.Name := 'fGutter';
  Self.fGutter.Size := System.Drawing.Size.Create(30, 230);
  Self.fGutter.TabIndex := 1;
//  Self.fGutter.Parent:= Self;
  //
  // TmwClientArea
  //
  Self.Controls.Add(Self.fGutter);
  Self.Controls.Add(Self.fView);
  Self.Name := 'TmwClientArea';
  Self.Size := System.Drawing.Size.Create(170, 230);
  Self.ResumeLayout(False);
end;
{$ENDREGION}

constructor TmwClientArea.Create;
begin
  fCreating:= True;
  inherited Create;
  Components:= nil;
  InitializeControl;
  //
  // TODO: Add any constructor code after InitializeControl call
  //
  fCreating:= False;
end;

procedure TmwClientArea.Dispose(Disposing: Boolean);
begin
  if Disposing then
  begin
    if Components <> nil then
      Components.Dispose();
  end;
  inherited Dispose(Disposing);
end;

function TmwClientArea.get_MainControl: TmwScrollingView;
begin
  Result:= (Self.Parent as TmwScrollingView);
end;

procedure TmwClientArea.OnResize(e: EventArgs);
begin
  case fCreating of
    False: Self.ReAlignChilds;
  end;
end;

procedure TmwClientArea.ReAlignChilds;
var
  MC: TmwScrollingView;
begin
  Self.SuspendLayout;
  MC:= Self.MainControl;
  //
  // fView
  //
  Self.fView.Left := MC.ScrollLeft;
  Self.fView.Top := MC.ScrollTop;
  Self.fView.Height := MC.ScrollHeight;
  Self.fView.Width := MC.ScrollWidth;
  //
  // fGutter
  //
  Self.fGutter.Top := MC.ScrollTop;
  Self.fGutter.Height := MC.ScrollHeight;

  MC.HScrollBar.LargeChange:= Self.Width - Self.Gutter.Width - MC.HScrollBar.SmallChange;
  MC.VScrollBar.LargeChange:= Self.Height - MC.VScrollBar.SmallChange;

  Self.ResumeLayout;
end;

{ TmwScrollingView }

constructor TmwScrollingView.Create;
begin
  fCreating:= True;
  inherited Create;
  Components:= nil;
  InitializeControl;
  //
  // TODO: Add any constructor code after InitializeControl call
  //
  fCreating:= False;
end;

procedure TmwScrollingView.Dispose(Disposing: Boolean);
begin
  if Disposing then
  begin
    if Components <> nil then
      Components.Dispose();
  end;
  inherited Dispose(Disposing);
end;

function TmwScrollingView.get_BorderHight: Integer;
begin
  Result:= 0;
  case Self.BorderStyle of
    TBorderStyle.Fixed3D: Result:= SystemInformation.Border3DSize.Height;
    TBorderStyle.FixedSingle: Result:= SystemInformation.BorderSize.Height;
  end;
end;

function TmwScrollingView.get_BorderWidth: Integer;
begin
  Result:= 0;
  case Self.BorderStyle of
    TBorderStyle.Fixed3D: Result:= SystemInformation.Border3DSize.Width;
    TBorderStyle.FixedSingle: Result:= SystemInformation.BorderSize.Width;
  end;
end;

function TmwScrollingView.get_ClientAreaHeight: Integer;
begin
  Result:= Self.ClientRectangle.Height;
  case Self.ScrollBars of
    TScrollBars.Both, TScrollBars.Horizontal:
      Result:= Self.ClientRectangle.Height - Self.HScrollBarHeight;
  end;
end;

function TmwScrollingView.get_ClientAreaWidth: Integer;
begin
  Result:= Self.ClientRectangle.Width;
  case Self.ScrollBars of
    TScrollBars.Both, TScrollBars.Vertical:
      Result:= Self.ClientRectangle.Width - Self.VScrollBarWidth;
  end;
end;

function TmwScrollingView.get_Gutter: TmwGutter;
begin
  Result:= fClientArea.Gutter;
end;

function TmwScrollingView.get_GutterWidth: Integer;
begin
  Result:= 0;
  if fClientArea <> nil then
    Result:= fClientArea.Gutter.Width;
end;

function TmwScrollingView.get_HScrollBarHeight: Integer;
begin
  Result:= 0;
  case Self.ScrollBars of
    TScrollBars.Both, TScrollBars.Horizontal:
      Result:= SystemInformation.HorizontalScrollBarHeight;
  end;
end;

function TmwScrollingView.get_HScrollBarVisible: Boolean;
begin
  Result:= False;
  case Self.ScrollBars of
    TScrollBars.Both, TScrollBars.Horizontal:
      Result:= True;
  end;
end;

function TmwScrollingView.get_HScrollBarWidth: Integer;
begin
  Result:= Self.ClientRectangle.Width - Self.VScrollBarWidth;
end;

function TmwScrollingView.get_ScrollHeight: Integer;
begin
  Result:= fClientArea.Height + fVScrollBar.Maximum;
end;

function TmwScrollingView.get_ScrollLeft: Integer;
begin
  Result:= fClientArea.Gutter.Width - fHScrollBar.Value;
end;

function TmwScrollingView.get_ScrollTop: Integer;
begin
  Result:= - fVScrollBar.Value;
end;

function TmwScrollingView.get_ScrollWidth: Integer;
begin
  Result:= fClientArea.Width + fHScrollBar.Maximum;
end;

function TmwScrollingView.get_View: TmwView;
begin
  Result:= fClientArea.View;
end;

function TmwScrollingView.get_VScrollBarHeight: Integer;
begin
  Result:= Self.ClientRectangle.Height - Self.HScrollBarHeight;
end;

function TmwScrollingView.get_VScrollBarVisible: Boolean;
begin
  Result:= False;
  case Self.ScrollBars of
    TScrollBars.Both, TScrollBars.Vertical:
      Result:= True;
  end;
end;

function TmwScrollingView.get_VScrollBarWidth: Integer;
begin
  Result:= 0;
  case Self.ScrollBars of
    TScrollBars.Both, TScrollBars.Vertical:
      Result:= SystemInformation.VerticalScrollBarWidth;
  end;
end;

{$REGION 'Initialization of TmwScrollingView'}
/// <summary>
/// </summary>
procedure TmwScrollingView.InitializeControl;
begin
  FScrollBars:= TScrollbars.Both;
  Self.SuspendLayout;
  Self.fClientArea := TmwClientArea.Create;
  Self.fVScrollBar := TmwVScrollBar.Create;
  Self.fHScrollBar := TmwHScrollBar.Create;
  //
  // fClientArea
  //
  Self.fClientArea.Anchor := (TAnchorStyles((((
    TAnchorStyles.Top
    or TAnchorStyles.Bottom)
    or TAnchorStyles.Left)
    or TAnchorStyles.Right)));
  Self.fClientArea.BackColor := System.Drawing.SystemColors.Control;
  Self.fClientArea.Location := System.Drawing.Point.Create(0, 0);
  Self.fClientArea.Name := 'fClientArea';
  Self.fClientArea.Size := System.Drawing.Size.Create(170, 230);
  Self.fClientArea.TabIndex := 0;
  Self.fClientArea.Parent:= Self;
  //
  // fVScrollBar
  // 
  Self.fVScrollBar.Anchor := (TAnchorStyles(((
    TAnchorStyles.Top
    or TAnchorStyles.Bottom)
    or TAnchorStyles.Right)));
  Self.fVScrollBar.Location := System.Drawing.Point.Create(170, 0);
  Self.fVScrollBar.Name := 'fVScrollBar';
  Self.fVScrollBar.Size := System.Drawing.Size.Create(15, 230);
  Self.fVScrollBar.TabIndex := 1;
  Self.fVScrollBar.Visible := VScrollBarVisible;
  //
  // fHScrollBar
  // 
  Self.fHScrollBar.Anchor := (TAnchorStyles((((
    TAnchorStyles.Bottom)
    or TAnchorStyles.Left)
    or TAnchorStyles.Right)));
  Self.fHScrollBar.Location := System.Drawing.Point.Create(0, 230);
  Self.fHScrollBar.Name := 'fHScrollBar';
  Self.fHScrollBar.Size := System.Drawing.Size.Create(170, 15);
  Self.fHScrollBar.TabIndex := 2;
  Self.fHScrollBar.Visible := HScrollBarVisible;
  //
  // TmwScrollingView
  // 
  Self.Controls.Add(Self.fHScrollBar);
  Self.Controls.Add(Self.fVScrollBar);
  Self.Controls.Add(Self.fClientArea);
  Self.Name := 'TmwScrollingView';
  Self.Size := System.Drawing.Size.Create(185, 245);
  Self.BorderStyle:= TBorderStyle.Fixed3D;
  Self.ResumeLayout(False);
end;
{$ENDREGION}

procedure TmwScrollingView.OnHScroll(e: ScrollEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnHScroll(Self, e);
  if Assigned(fHScroll) then
    fHScroll(Self, e);
end;

procedure TmwScrollingView.OnHValueChanged(e: EventArgs);
begin
  if fCommander <> nil then
    fCommander.OnHValueChanged(Self, e);
  if Assigned(fHValueChanged) then
    fHValueChanged(Self, e);
end;

procedure TmwScrollingView.OnKeyDown(e: KeyEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnKeyDown(Self, e);
  inherited OnKeyDown(e);
end;

procedure TmwScrollingView.OnKeyPress(e: KeyPressEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnKeyPress(Self, e);
  inherited OnKeyPress(e);
end;

procedure TmwScrollingView.OnKeyUp(e: KeyEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnKeyUp(Self, e);
  inherited OnKeyUp(e);
end;

procedure TmwScrollingView.OnMouseDown(e: MouseEventArgs);
begin
  Self.Focus;
  inherited OnMouseDown(e);
end;

procedure TmwScrollingView.OnMouseDownGutter(e: MouseEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnMouseDownGutter(Self, e);
  if Assigned(fMouseDownGutter) then
    fMouseDownGutter(Self, e);
end;

procedure TmwScrollingView.OnMouseDownView(e: MouseEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnMouseDown(Self, e);
  if Assigned(fMouseDownView) then
    fMouseDownView(Self, e);
end;

procedure TmwScrollingView.OnMouseEnterView(e: EventArgs);
begin
  if fCommander <> nil then
    fCommander.OnMouseEnter(Self, e);
  if Assigned(fMouseEnterView) then
    fMouseEnterView(Self, e);
end;

procedure TmwScrollingView.OnMouseHoverView(e: EventArgs);
begin
  if fCommander <> nil then
    fCommander.OnMouseHover(Self, e);
  if Assigned(fMouseHoverView) then
    fMouseHoverView(Self, e);
end;

procedure TmwScrollingView.OnMouseLeaveView(e: EventArgs);
begin
  if fCommander <> nil then
    fCommander.OnMouseLeave(Self, e);
  if Assigned(fMouseLeaveView) then
    fMouseLeaveView(Self, e);
end;

procedure TmwScrollingView.OnMouseMoveView(e: MouseEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnMouseMove(Self, e);
  if Assigned(fMouseMoveView) then
    fMouseMoveView(Self, e);
end;

procedure TmwScrollingView.OnMouseUpView(e: MouseEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnMouseUp(Self, e);
  if Assigned(fMouseUpView) then
    fMouseUpView(Self, e);
end;

procedure TmwScrollingView.OnMouseWheelView(e: MouseEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnMouseWheel(Self, e);
  if Assigned(fMouseWheelView) then
    fMouseWheelView(Self, e);
end;

procedure TmwScrollingView.OnPaintGutter(e: PaintEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnPaintGutter(Self, e);
  if Assigned(fPaintGutter) then
    fPaintGutter(Self, e);
end;

procedure TmwScrollingView.OnPaintView(e: PaintEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnPaint(Self, e);
  if Assigned(fPaintView) then
    fPaintView(Self, e);
end;

procedure TmwScrollingView.OnResize(e: EventArgs);
begin
  case fCreating of
    False:
      begin
        case fCommander <> nil of
          True: fCommander.OnResize(Self, e);
          False: Self.ReAlignChilds;
        end;
      end;
  end;
  inherited OnResize(e);
end;

procedure TmwScrollingView.OnVScroll(e: ScrollEventArgs);
begin
  if fCommander <> nil then
    fCommander.OnVScroll(Self, e);
  if Assigned(fVScroll) then
    fVScroll(Self, e);
end;

procedure TmwScrollingView.OnVValueChanged(e: EventArgs);
begin
  if fCommander <> nil then
    fCommander.OnVValueChanged(Self, e);
  if Assigned(fVValueChanged) then
    fVValueChanged(Self, e);
end;

procedure TmwScrollingView.ReAlignChilds;
begin
  Self.SuspendLayout;
  //
  // fVScrollBar
  //
  Self.fVScrollBar.Left := ClientAreaWidth;
  Self.fVScrollBar.Height := ClientAreaHeight;
  Self.fVScrollBar.Width := VScrollBarWidth;
  Self.fVScrollBar.Visible := VScrollBarVisible;
  //
  // fHScrollBar
  //
  Self.fHScrollBar.Top := ClientAreaHeight;
  Self.fHScrollBar.Height := HScrollBarHeight;
  Self.fHScrollBar.Width := ClientAreaWidth;
  Self.fHScrollBar.Visible := HScrollBarVisible;
  //
  // fClientArea
  //
  Self.fClientArea.Height := ClientAreaHeight;
  Self.fClientArea.Width := ClientAreaWidth;

  Self.ResumeLayout;
end;

procedure TmwScrollingView.set_Commander(const Value: TmwScrollingViewCommander);
begin
  fCommander := Value;
end;

procedure TmwScrollingView.set_GutterWidth(const Value: Integer);
begin
  if GutterWidth <> Value then
  begin
    if fClientArea <> nil then
      fClientArea.Gutter.Width := Value;
    fClientArea.ReAlignChilds;
  end;
end;

procedure TmwScrollingView.set_HScroll(const Value: ScrollEventHandler);
begin
  fHScroll := Value;
end;

procedure TmwScrollingView.set_HValueChanged(const Value: EventHandler);
begin
  fHValueChanged := Value;
end;

procedure TmwScrollingView.set_MouseDownGutter(
  const Value: MouseEventHandler);
begin
  fMouseDownGutter := Value;
end;

procedure TmwScrollingView.set_MouseDownView(
  const Value: MouseEventHandler);
begin
  fMouseDownView := Value;
end;

procedure TmwScrollingView.set_MouseEnterView(const Value: EventHandler);
begin
  fMouseEnterView := Value;
end;

procedure TmwScrollingView.set_MouseHoverView(const Value: EventHandler);
begin
  fMouseHoverView := Value;
end;

procedure TmwScrollingView.set_MouseLeaveView(const Value: EventHandler);
begin
  fMouseLeaveView := Value;
end;

procedure TmwScrollingView.set_MouseMoveView(
  const Value: MouseEventHandler);
begin
  fMouseMoveView := Value;
end;

procedure TmwScrollingView.set_MouseUpView(const Value: MouseEventHandler);
begin
  fMouseUpView := Value;
end;

procedure TmwScrollingView.set_MouseWheelView(
  const Value: MouseEventHandler);
begin
  fMouseWheelView := Value;
end;

procedure TmwScrollingView.set_PaintGutter(const Value: PaintEventHandler);
begin
  fPaintGutter := Value;
end;

procedure TmwScrollingView.set_PaintView(const Value: PaintEventHandler);
begin
  fPaintView := Value;
end;

procedure TmwScrollingView.set_ScrollBars(const Value: TScrollBars);
begin
  if FScrollBars <> Value then
  begin
    FScrollBars := Value;
    ReAlignChilds;
  end;
end;

procedure TmwScrollingView.set_VScroll(const Value: ScrollEventHandler);
begin
  fVScroll := Value;
end;

procedure TmwScrollingView.set_VValueChanged(const Value: EventHandler);
begin
  fHValueChanged := Value;
end;

end.
